from instrControl import *
import win32com.client as win32
import Range
from arrayController import *
from analogDiscover import *
import time

# this range  integrates the NSI moter contolloers, LAN communication with the PNA, and the MURI Array controller
# used for testing code
class NSIRange(Range.Range):
    # address and coordinate list are of tiles in the array
    # tiles are programmed and optimized in the order given by address list
    # the FIRST tile in the addressList is the tile that is as a for phase optimization reference
    # phase accuracy is the "maximum" error of the array phase drivess
    def __init__(self, addressList, coordinateList, state_mapper = None, phase_accuracy=1, ifbw=600000,verbose=False, debug=False):
        self.verbose = verbose
        self.phase_accuracy = phase_accuracy
        self.debug = debug
        self.state_mapper = state_mapper
        #setup motor controller
        print "-----------INITIALIZING RANGE-----------"
        self.nsiServer = win32.Dispatch("NSI2000.Server")
        self.nsiController = self.nsiServer.AppConnection.SRObject
        self.nsiController.CONTROLLER_AUX_AXIS_SET("SNPol")

        if self.debug:
            self.supply_meter = Keysight_34465A.Keysight_34465A('USB0::0x2A8D::0x1301::MY53200235::0::INSTR')
            self.temp_meter = Keysight_34465A.Keysight_34465A('USB0::0x2A8D::0x0101::MY54500181::0::INSTR')
            self.temp_meter.prepDCVoltageMeasurement('100 mV')
            self.supply_meter.prepDCCurrentMeasurement('3')

            self.field_fox = FieldFox.FieldFox('TCPIP1::A-N9918A-04851::inst0::INSTR')
            self.field_fox.prepSpectrumMeasurement( 2.5*10**9, 40*10**6, 30*10**3, 30*10**3)
            self.field_fox.prepSpectrumMeasurement( 2.5*10**9, 40*10**6, 30*10**3, 30*10**3)

        print "Returning to sweep origin..."
        self.moveToTheta(0)
        self.moveToPhi(0)
        print "-----------RANGE INITIALIZED-----------\n"

        print "-----------INITIALIZING VNA-----------"
        # setup vna
        self.vna = Agilent_N5230A.Agilent_N5230A('AGILENT-5CFE7DF')
        self.vna.prepRangeMeasurement(2.5,ifbw)
        print "-----------VNA INITIALIZED -----------\n"

        print('USER: Click OKAY on NSI app error to disconnect NSI from VNA.')
        a = raw_input('Press Enter when complete to continue:')

        # set up array
        print "-----------INITIALIZING ARRAY-----------"
        ss_pin = 2
        clk_pin = 0
        mosi_pin = 1
        miso_pin = 3

        self.spi_channel = spiController_v2.spiController_v2(mosi_pin, miso_pin, clk_pin, ss_pin)

        broadcast_addr = 234
        outputPower = 8
        self.array = muriArrayV2.muriArrayV2(self.spi_channel, addressList, coordinateList, broadcast_addr, state_mapper = state_mapper)
        print "Turning off Metagaps..."
        self.array.arrayDisableMetagaps()
        print "Setting output power..."
        print "Normalizing Tile Output Powers..."
        self.array.normalizeTileOutputPower()
        print "Turning on RF..."
        self.array.arrayEnableRF()
        print "-----------ARRAY INITIALIZED-----------\n"

    def __del__(self):
        print "Turning off Metagaps..."
        self.array.arrayDisableMetagaps()
        print "Returning to sweep origin..."
        self.moveToTheta(0)
        self.moveToPhi(0)

    def moveToTheta(self, theta):
        self.nsiController.MOVE_HAXIS(str(theta) + " deg")
        if self.verbose:
            print "moving to position", theta

    def moveToPhi(self, phi):
        self.nsiController.MOVE_VAXIS(str(phi) + " deg")
        self.nsiController.CONTROLLER_AUX_AXIS_MOVE(str(-1*phi) + " deg")
        if self.verbose:
            print "moving to position", phi

    def programState(self,state):
        if self.verbose:
            print "programming state: ",state
        if self.state_mapper is None:
            self.array.programArrayMetaGaps(state,broadcast = True)
        else:
            sheet_states_e, sheet_states_h = self.array.convertStateToSheetData(state)
            self.array.programArrayMetaGaps(sheet_states_e,mg_data_h=sheet_states_h)

    def programPhases(self,phases):
        if self.verbose:
            print "programming phases: ",phases
        self.array.programArrayPhase(phases)

    def measure(self):
        #time.sleep(0.5)
        return self.vna.measure()

    def measureTemp(self):
        return 1000*self.temp_meter.measure()

    def measureSupply(self):
        return self.supply_meter.measure()

    def measureSpectrum(self):
        return self.field_fox.getSpectrum()

    def fastMeasure(self):
        self.vna.fastMeasure()

    def pullData(self):
        return self.vna.pullData()

    # phases are returned in order of address list
    def optimizePhase(self):
        if self.verbose:
            print "Optimizing Phases"
        # Turn off all tiles
        self.array.arrayDisableRF()
        optimum_phases = [0]
        referenceTile = self.array.getTileByAddress(self.array.addressList[0])
        referenceTile.enableRF()
        for i in range(1,self.array.numTiles):
            tile = self.array.getTileByAddress(self.array.addressList[i])
            optimum_phases += [self.searchForOptPhase(tile)]
        return optimum_phases

    def searchForOptPhase(self,tile):
            test_phase = 0
            tile.programPhase(test_phase,rfEnable = True)
            meas_0 = self.measure()
            test_phase = 120
            tile.programPhase(test_phase)
            meas_120 = self.measure()
            test_phase = 240
            tile.programPhase(test_phase)
            meas_240 = self.measure()

            min_initial_meas = min([meas_0,meas_120,meas_240])
            # first third
            if min_initial_meas == meas_240:
                left_phase = 0
                left_meas = meas_0
                right_phase = 120
                right_meas = meas_120
            # second third
            if min_initial_meas == meas_0:
                left_phase = 120
                left_meas = meas_120
                right_phase = 240
                right_meas = meas_240
            # third third
            if min_initial_meas == meas_120:
                left_phase = 240
                left_meas = meas_240
                right_phase = 360
                right_meas = meas_0
            return self.binarySearchForOptPhase(tile,left_phase,left_meas,right_phase,right_meas)

    def binarySearchForOptPhase(self,tile,left_phase,left_meas,right_phase,right_meas):
        new_phase = (left_phase+right_phase)/2.0
        phase_delta = right_phase - left_phase
        # if the difference between endpoints is less than twice the accuracy tolerance
        # then the midpoint will always be at most the accuracy tolerance away from the maximum
        if phase_delta < 2*self.phase_accuracy:
            return new_phase
        tile.programPhase(new_phase)
        # new meas WILL be greater that sides due to sinusoidal nature (ignoring noise)
        new_meas = self.measure()
        # replace the minimum with the new phase
        if left_meas > right_meas:
            return self.binarySearchForOptPhase(tile,left_phase,left_meas,new_phase,new_meas)
        else:
            return self.binarySearchForOptPhase(tile,new_phase,new_meas,right_phase,right_meas)
